<?php
/*
Plugin Name: Model Manager
Description: Manage AI models as part of the LucyDream plugin.
Version: 1.0
Author: LucyDream
*/

if (!defined('ABSPATH')) {
    exit;
}

define('MM_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MM_PLUGIN_URL', plugin_dir_url(__FILE__));

require_once MM_PLUGIN_DIR . 'add-edit-model.php';
require_once MM_PLUGIN_DIR . 'prompt-settings.php';

// Activation hook to create tables
register_activation_hook(__FILE__, 'mm_install');
function mm_install() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    $models_table = $wpdb->prefix . 'model_manager_models';
    $sql_models = "CREATE TABLE $models_table (
    id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    tune_id VARCHAR(20) DEFAULT NULL,
    type VARCHAR(50) NOT NULL,
    spicy_credit_cost DECIMAL(5,2) NOT NULL DEFAULT 0.50,
    trigger_name VARCHAR(100) DEFAULT NULL,
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    PRIMARY KEY (id),
    UNIQUE KEY tune_id (tune_id)
) $charset_collate;";

    $user_models_table = $wpdb->prefix . 'model_manager_user_models';
    $sql_user_models = "CREATE TABLE $user_models_table (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        name VARCHAR(100) NOT NULL,
        tune_id VARCHAR(20) NOT NULL,
        user_id BIGINT(20) UNSIGNED NOT NULL,
        class_name VARCHAR(10) NOT NULL,
        spicy_credit_cost DECIMAL(5,2) NOT NULL DEFAULT 0.50,
        is_active TINYINT(1) NOT NULL DEFAULT 1,
        training_status VARCHAR(20) DEFAULT 'pending',
        created_at DATETIME DEFAULT NULL,
        full_name VARCHAR(200) DEFAULT NULL,
        safetensors_path TEXT DEFAULT NULL,
        safetensors_expires_at DATETIME DEFAULT NULL,
        cost_overridden TINYINT(1) NOT NULL DEFAULT 0,
        PRIMARY KEY (id),
        UNIQUE KEY tune_id (tune_id)
    ) $charset_collate;";

    // Create prompt limits table
    $prompt_limits_table = $wpdb->prefix . 'model_manager_prompt_limits';
    $sql_prompt_limits = "CREATE TABLE IF NOT EXISTS $prompt_limits_table (
    id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
    user_id BIGINT(20) UNSIGNED NOT NULL,
    prompt_type ENUM('spicy') NOT NULL,
    free_limit INT UNSIGNED NOT NULL DEFAULT 0,
    used_prompts INT UNSIGNED NOT NULL DEFAULT 0,
    price_override DECIMAL(5,2) DEFAULT NULL,
    last_updated DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    UNIQUE KEY user_prompt_type (user_id, prompt_type)
) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql_models);
    dbDelta($sql_user_models);
    dbDelta($sql_prompt_limits);

    // Set default prompt generation settings
    if (!get_option('mm_spicy_prompt_price')) {
        update_option('mm_spicy_prompt_price', 0.05); // Default Spicy prompt price
    }
    if (!get_option('mm_spicy_prompt_free_limit')) {
        update_option('mm_spicy_prompt_free_limit', 3); // Default Spicy free limit
    }


    // Set default model creation cost
    if (!get_option('mm_model_creation_cost')) {
        update_option('mm_model_creation_cost', 6.00); // Default model creation cost in credits
    }

    // Seed initial models with LoRA tags and trigger names
    $initial_models = [
        ['name' => 'Lucy Dream', 'tune_id' => '2195981', 'type' => 'default', 'spicy_credit_cost' => 0.50, 'trigger_name' => 'blonde woman with green eyes'],
        ['name' => 'My Models', 'tune_id' => NULL, 'type' => 'user_container', 'spicy_credit_cost' => 0.50, 'trigger_name' => NULL],
    ];
    foreach ($initial_models as $model) {
        $exists = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $models_table WHERE name = %s", $model['name']));
        if (!$exists) {
            $wpdb->insert($models_table, $model);
        }
    }
}

// Model Management Functions
function mm_get_all_models($active_only = true) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_models';
    $where = $active_only ? 'WHERE is_active = 1' : '';
    return $wpdb->get_results("SELECT * FROM $table_name $where", ARRAY_A);
}

function mm_add_model($data) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_models';
    return $wpdb->insert($table_name, [
        'name' => sanitize_text_field($data['name']),
        'tune_id' => sanitize_text_field($data['tune_id']),
        'type' => sanitize_text_field($data['type']),
        'spicy_credit_cost' => floatval($data['spicy_credit_cost']),
        'trigger_name' => sanitize_text_field($data['trigger_name']),
    ]);
}

function mm_update_model($id, $data) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_models';
    $user_models_table = $wpdb->prefix . 'model_manager_user_models';

    // Update the model in model_manager_models
    $result = $wpdb->update($table_name, [
        'name' => sanitize_text_field($data['name']),
        'tune_id' => sanitize_text_field($data['tune_id']),
        'type' => sanitize_text_field($data['type']),
        'spicy_credit_cost' => floatval($data['spicy_credit_cost']),
        'trigger_name' => sanitize_text_field($data['trigger_name']),
    ], ['id' => $id]);

    if ($data['type'] === 'user_container') {
        $new_cost = floatval($data['spicy_credit_cost']);
        $wpdb->update(
            $user_models_table,
            [
                'spicy_credit_cost' => $new_cost,
                'cost_overridden' => 0 
            ],
            ['is_active' => 1], 
            ['%f', '%d'],
            ['%d']
        );
    }

    return $result;
}

function mm_toggle_model_status($id, $status) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_models';
    return $wpdb->update($table_name, ['is_active' => $status], ['id' => $id]);
}

// User Model Management Functions
function mm_add_user_model($data) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    $models_table = $wpdb->prefix . 'model_manager_models';
    $default_cost = $wpdb->get_var("SELECT spicy_credit_cost FROM $models_table WHERE type = 'user_container' LIMIT 1") ?? 0.50;
    $cost_overridden = 0;
    if (isset($data['spicy_credit_cost'])) {
        $provided_cost = floatval($data['spicy_credit_cost']);
        if ($provided_cost != 0.50) {
            $cost_overridden = ($provided_cost != $default_cost) ? 1 : 0;
        } else {
            $provided_cost = $default_cost;
        }
    } else {
        $provided_cost = $default_cost;
    }
    $data['spicy_credit_cost'] = $provided_cost;
    return $wpdb->insert($table_name, [
        'name' => sanitize_text_field($data['name']),
        'tune_id' => sanitize_text_field($data['tune_id']),
        'user_id' => intval($data['user_id']),
        'class_name' => sanitize_text_field($data['class_name']),
        'spicy_credit_cost' => floatval($data['spicy_credit_cost']),
        'is_active' => 1, // Fixed to active since no edit
        'training_status' => isset($data['training_status']) ? $data['training_status'] : 'pending',
        'created_at' => current_time('mysql'),
        'full_name' => sanitize_text_field($data['full_name']),
        'safetensors_path' => isset($data['safetensors_path']) ? sanitize_text_field($data['safetensors_path']) : null,
        'safetensors_expires_at' => isset($data['safetensors_expires_at']) ? $data['safetensors_expires_at'] : null,
        'cost_overridden' => $cost_overridden
    ]);
}

function mm_update_user_model($id, $data) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    return $wpdb->update($table_name, [
        'name' => sanitize_text_field($data['name']),
        'tune_id' => sanitize_text_field($data['tune_id']),
        'user_id' => intval($data['user_id']),
        'class_name' => sanitize_text_field($data['class_name']),
        'spicy_credit_cost' => floatval($data['spicy_credit_cost']),
        // No is_active update
    ], ['id' => $id]);
}

function mm_get_user_models($user_id = null, $active_only = true) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    $where = $active_only ? 'WHERE is_active = 1' : '';
    if ($user_id) {
        $where .= $where ? " AND user_id = %d" : "WHERE user_id = %d";
        return $wpdb->get_results($wpdb->prepare("SELECT id, name, tune_id, user_id, class_name, spicy_credit_cost, is_active, training_status, created_at, full_name, safetensors_path, safetensors_expires_at FROM $table_name $where", $user_id), ARRAY_A);
    }
    return $wpdb->get_results("SELECT id, name, tune_id, user_id, class_name, spicy_credit_cost, is_active, training_status, created_at FROM $table_name $where", ARRAY_A);
}

function mm_toggle_user_model_status($id, $status) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    return $wpdb->update($table_name, ['is_active' => $status], ['id' => $id]);
}

function mm_delete_user_model($id) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    return $wpdb->delete($table_name, ['id' => $id]);
}

// Admin Interface
add_action('admin_menu', 'mm_admin_menu');
function mm_admin_menu() {
    add_menu_page('Model Manager', 'Model Manager', 'manage_options', 'model-manager', 'mm_admin_page', 'dashicons-format-gallery', 81);
    add_submenu_page('model-manager', 'Add/Edit Model', 'Add/Edit Model', 'manage_options', 'model-manager-add-edit', 'mm_add_edit_model_page');
    add_submenu_page('model-manager', 'Prompt Settings', 'Prompt Settings', 'manage_options', 'model-manager-prompt-settings', 'mm_prompt_settings_page');
}

function mm_admin_page() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }

    global $wpdb;
    $models_table = $wpdb->prefix . 'model_manager_models';
    $user_models_table = $wpdb->prefix . 'model_manager_user_models';

    // Handle model creation cost update
    if (isset($_POST['mm_update_creation_cost'])) {
        if (isset($_POST['mm_model_creation_cost'])) {
            update_option('mm_model_creation_cost', floatval($_POST['mm_model_creation_cost']));
            echo '<div class="notice notice-success"><p>Model creation cost updated!</p></div>';
        }
    }

    if (isset($_GET['action']) && $_GET['action'] === 'toggle_model' && isset($_GET['id'])) {
        $id = intval($_GET['id']);
        $status = intval($_GET['status']);
        mm_toggle_model_status($id, $status);
        echo '<div class="notice notice-success"><p>Model status updated!</p></div>';
    }

    // Handle user model delete (no toggle)
    if (isset($_GET['action']) && $_GET['action'] === 'delete_user_model' && isset($_GET['id'])) {
        mm_delete_user_model(intval($_GET['id']));
        echo '<div class="notice notice-success"><p>User model deleted!</p></div>';
    }

    $models = mm_get_all_models(false);
    $user_models = mm_get_user_models(null, false);
    $filter_user_id = isset($_GET['filter_user_id']) ? intval($_GET['filter_user_id']) : '';
    if ($filter_user_id) {
        $user_models = mm_get_user_models($filter_user_id, false);
    }
    ?>
    <div class="wrap">
        <h1>Model Manager</h1>

        <!-- Model Creation Cost Section -->
        <h2>Model Creation Cost</h2>
        <form method="post">
            <input type="hidden" name="mm_update_creation_cost" value="1">
            <label for="mm_model_creation_cost" style="font-size: 16px;">Cost in credits to create a model:</label>
            <input type="number" step="0.01" name="mm_model_creation_cost" id="mm_model_creation_cost" value="<?php echo esc_attr(get_option('mm_model_creation_cost', 10.00)); ?>">
            <input type="submit" value="Save" class="button">
        </form>

        <!-- Default Models -->
        <h2 style="margin-top: 40px;">Default Models</h2>
        <table class="wp-list-table widefat fixed striped">
            <thead><tr><th>Name</th><th>Tune ID</th><th>Type</th><th>Spicy Cost</th><th>Trigger Name</th><th>Status</th><th>Actions</th></tr></thead>
            <tbody>
                <?php foreach ($models as $model): ?>
                    <tr>
                        <td><?php echo esc_html($model['name']); ?></td>
                        <td><?php echo esc_html($model['tune_id'] ?: 'N/A'); ?></td>
                        <td><?php echo esc_html($model['type']); ?></td>
                        <td><?php echo esc_html($model['spicy_credit_cost']); ?></td>
                        <td><?php echo esc_html($model['trigger_name'] ?: 'N/A'); ?></td>
                        <td><?php echo $model['is_active'] ? 'Active' : 'Inactive'; ?></td>
                        <td>
                            <a href="<?php echo admin_url('admin.php?page=model-manager-add-edit&action=edit&model_type=default&id=' . $model['id']); ?>">Edit</a> 
                            <a href="<?php echo admin_url('admin.php?page=model-manager&action=toggle_model&id=' . $model['id'] . '&status=' . ($model['is_active'] ? 0 : 1)); ?>" ></a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <!-- User Models (My Models) -->
        <h2 style="margin-top: 40px;">User Models (My Models)</h2>
        <form method="get">
            <input type="hidden" name="page" value="model-manager">
            <label for="filter_user_id">Filter by User ID:</label>
            <input type="number" name="filter_user_id" id="filter_user_id" value="<?php echo esc_attr($filter_user_id); ?>">
            <input type="submit" value="Filter" class="button">
        </form>
        <table class="wp-list-table widefat fixed striped">
            <thead><tr><th>Name</th><th>Tune ID</th><th>User ID</th><th>Trigger Name</th><th>Spicy Cost</th><th>Status</th><th>Actions</th></tr></thead>
            <tbody>
                <?php foreach ($user_models as $model): ?>
                    <tr>
                        <td><?php echo esc_html($model['name']); ?></td>
                        <td><?php echo esc_html($model['tune_id']); ?></td>
                        <td><?php echo esc_html($model['user_id']); ?></td>
                        <td><?php echo esc_html($model['class_name']); ?></td>
                        <td><?php echo esc_html($model['spicy_credit_cost']); ?></td>
                        <td><?php echo $model['is_active'] ? 'Active' : 'Inactive'; ?></td>
                        <td>
                            <a href="<?php echo admin_url('admin.php?page=model-manager-add-edit&action=edit&model_type=user&id=' . $model['id']); ?>">Edit</a> |
                            <a href="<?php echo admin_url('admin.php?page=model-manager&action=delete_user_model&id=' . $model['id']); ?>" onclick="return confirm('Are you sure?');">Delete</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php
}

// Frontend Integration
add_filter('mm_model_options', 'mm_generate_model_options');
function mm_generate_model_options($options = '') {
    $models = mm_get_all_models();
    $options = '<option value="" selected disabled>---</option>';
    foreach ($models as $model) {
        $display_name = $model['name'];
        $value = $model['tune_id'] ?: 'my_models';
        $options .= "<option value='$value'>$display_name</option>";
    }
    return $options;
}