<?php
/**
 * File: spicy-pending-tasks.php
 * Description: Handles Spicy pending tasks for the LucyDream Image Webhook plugin.
 * Version: 1.0
 * Author: LucyDream
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Ensure required dependencies are available
global $wpdb;

// Register REST route for Spicy pending tasks
add_action('rest_api_init', function () {
    register_rest_route('lucydream/v1', '/check-pending-spicy-tasks', [
        'methods' => 'POST',
        'callback' => 'check_pending_spicy_tasks',
        'permission_callback' => function ($request) {
            $secret = 'Ue9uF_vARm.a7zW!';
            return isset($request['secret']) && $request['secret'] === $secret;
        }
    ]);
});

// Check pending Spicy tasks
function check_pending_spicy_tasks($request) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'spicy_pending_tasks';
    $tasks = $wpdb->get_results("SELECT * FROM $table_name WHERE status = 'pending' AND task_id NOT LIKE 'pending_%' LIMIT 50", ARRAY_A);

    if ($wpdb->last_error) {
        return new WP_REST_Response(['error' => 'Database error'], 500);
    }

    foreach ($tasks as $task) {
        $task_id = $task['task_id'];
        $user_id = $task['user_id'];
        $tune_id = $task['tune_id'];
        $full_prompt = $task['prompt'];
        $display_user_prompt = $task['user_prompt'];
        $pending_transaction_id = $task['pending_transaction_id'];

        $model_tune_ids = ['2195981' => 'default', '' => 'alternative', '' => 'premium1'];
        $model = array_search($tune_id, $model_tune_ids) ? $model_tune_ids[$tune_id] : ($tune_id ? 'my_models' : 'default');
        $is_user_model = $model === 'my_models';
        $premium_models = ['premium1'];
        $credit_cost = $is_user_model ? 0.50 : (in_array($model, $premium_models) ? 0.7 : 0.5);

        $response = wp_remote_get(XXX_API_URL . "/task_status/$task_id", [
            'headers' => ['Authorization' => 'Bearer ' . XXX_AUTH_TOKEN],
            'timeout' => 5
        ]);

        if (is_wp_error($response)) {
            continue;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($body['status'])) {
            continue;
        }

        if ($body['status'] === 'COMPLETED' && !empty($body['image_urls'])) {
            $image_url = $body['image_urls'][0];

            $image_response = wp_remote_get($image_url, ['timeout' => 10]);
            if (is_wp_error($image_response)) {
                $wpdb->update($table_name, ['status' => 'failed', 'updated_at' => current_time('mysql')], ['task_id' => $task_id]);
                lucydream_revert_pending_credits($user_id, $credit_cost, $pending_transaction_id, "Image fetch failed for Spicy task $task_id");
                continue;
            }
            $image_data = wp_remote_retrieve_body($image_response);

            $random_key = bin2hex(random_bytes(16));
            $upload_dir = wp_upload_dir();
            $user_dir = "/lucydream-images/$user_id/$tune_id/";
            $file_path = $upload_dir['basedir'] . $user_dir . $random_key . '_' . $tune_id . '.jpg';
            $thumbnail_path = $upload_dir['basedir'] . $user_dir . $random_key . '_' . $tune_id . '_thumb.jpg';
            $file_url = $upload_dir['baseurl'] . $user_dir . $random_key . '_' . $tune_id . '.jpg';
            $thumbnail_url = $upload_dir['baseurl'] . $user_dir . $random_key . '_' . $tune_id . '_thumb.jpg';

            if (!file_exists(dirname($file_path))) {
                mkdir(dirname($file_path), 0755, true);
            }
            if (file_put_contents($file_path, $image_data) === false) {
                $wpdb->update($table_name, ['status' => 'failed', 'updated_at' => current_time('mysql')], ['task_id' => $task_id]);
                lucydream_revert_pending_credits($user_id, $credit_cost, $pending_transaction_id, "Image storage failed for Spicy task $task_id");
                continue;
            }
            chmod($file_path, 0644);

            $thumbnail_generated = generate_thumbnail($file_path, $thumbnail_path);
            if (!$thumbnail_generated) {
                $thumbnail_url = $file_url;
            }
            enforce_image_limit(100, $user_id, $tune_id);

            update_option("lucydream_image_user_{$random_key}", $user_id);
            update_option("lucydream_image_prompt_{$random_key}", $full_prompt);
            update_option("lucydream_image_user_prompt_{$random_key}", $display_user_prompt);
            update_option("lucydream_image_thumbnail_{$random_key}", $thumbnail_url);

            $commit_response = lucydream_commit_pending_credits($user_id, $credit_cost, $pending_transaction_id, 'Spicy Image generation completed');
            if (is_wp_error($commit_response) || !$commit_response['success']) {
                $error_msg = is_wp_error($commit_response) ? $commit_response->get_error_message() : $commit_response['message'];
                lucydream_revert_pending_credits($user_id, $credit_cost, $pending_transaction_id, 'Failed to commit credits');
                $wpdb->update($table_name, ['status' => 'failed', 'updated_at' => current_time('mysql')], ['task_id' => $task_id]);
                continue;
            }

            $wpdb->update($table_name, ['status' => 'completed', 'updated_at' => current_time('mysql')], ['task_id' => $task_id]);
        } elseif ($body['status'] === 'FAILED') {
            lucydream_revert_pending_credits($user_id, $credit_cost, $pending_transaction_id, 'Spicy Image generation failed');
            $wpdb->update($table_name, ['status' => 'failed', 'updated_at' => current_time('mysql')], ['task_id' => $task_id]);
        } else {
            $wpdb->update($table_name, ['updated_at' => current_time('mysql')], ['task_id' => $task_id]);
        }
    }

    return new WP_REST_Response(['success' => true, 'message' => 'Checked Spicy pending tasks'], 200);
}

// Create Spicy pending tasks table
function create_spicy_pending_tasks_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'spicy_pending_tasks';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        task_id VARCHAR(255) NOT NULL,
        user_id BIGINT(20) UNSIGNED NOT NULL,
        tune_id VARCHAR(255) NOT NULL,
        prompt TEXT NOT NULL,
        user_prompt TEXT NOT NULL,
        created_at DATETIME NOT NULL,
        updated_at DATETIME NOT NULL,
        status ENUM('pending', 'completed', 'failed') DEFAULT 'pending',
        pending_transaction_id BIGINT(20) UNSIGNED DEFAULT NULL
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);

    if ($wpdb->last_error) {
    } else {
    }
}

// Call the table creation function on plugin activation
register_activation_hook(__FILE__, 'create_spicy_pending_tasks_table');

?>