<?php

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Save a Spicy-generated prompt to the database.
 **/
function save_spicy_prompt($request) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'spicy_prompts';
    $prompt = sanitize_text_field($request['prompt']);

    $result = $wpdb->insert(
        $table_name,
        array(
            'prompt' => $prompt,
            'created_at' => current_time('mysql')
        ),
        array('%s', '%s')
    );

    if ($result === false) {
        return new WP_REST_Response(array('error' => 'Failed to save prompt'), 500);
    }

    return new WP_REST_Response(array('success' => true), 200);
}

/**
 * Generate a new Spicy prompt.
 *
 * @param WP_REST_Request $request The REST request.
 * @return WP_REST_Response|WP_Error Response containing the prompt or error.
 */
function generate_spicy_prompt($request) {
    $proxy_url = defined('PROXY_URL') ? PROXY_URL : '';
    $proxy_api_key = defined('PROXY_SECRET_KEY') ? PROXY_SECRET_KEY : '';

    if (empty($proxy_url) || empty($proxy_api_key)) {
        return new WP_Error('config_error', 'Server configuration issue', array('status' => 500));
    }

    // Check prompt limit and balance
    $user_id = get_current_user_id();
    $prompt_type = 'spicy';
    $limit_data = mm_get_prompt_limit($user_id, $prompt_type);
    $balance = lucydream_calculate_balance($user_id);
    $price = $limit_data['price'];
    $free_limit = $limit_data['free_limit'];
    $used_prompts = $limit_data['used_prompts'];

    if ($free_limit > 0 && $used_prompts >= $free_limit) {
        if ($balance < $price) {
            return new WP_Error('insufficient_balance', "Insufficient credits for $prompt_type prompt.", array('status' => 402));
        }
        $pending_response = lucydream_hold_credits($user_id, $price, "Prompt generation ($prompt_type)");
        if (is_wp_error($pending_response) || !$pending_response['success']) {
            return new WP_Error('credit_error', 'Failed to hold credits.', array('status' => 500));
        }
        $transaction_id = $pending_response['transaction_id'];
    }

    $proxy_endpoint = $proxy_url . 'wp-json/whitelabel/v1/generate-prompt';
    $response = wp_remote_post($proxy_endpoint, array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Proxy-Key' => $proxy_api_key,
            'X-Request-Type'  => 'api',
        ),
        'body' => json_encode(array('api_mode' => 'spicy')),
        'timeout' => 20
    ));

    if (is_wp_error($response)) {
        return new WP_Error('proxy_error', 'Failed to generate prompt. Please try again.', array('status' => 500));
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);
    $status_code = wp_remote_retrieve_response_code($response);
    $raw_body = wp_remote_retrieve_body($response);
    $data = json_decode($raw_body, true);

    if ($status_code !== 200 || empty($data['prompt'])) {
        return new WP_Error('proxy_error', $data['message'] ?? 'Prompt generation failed.', array('status' => $status_code));
    }

    $prompt = sanitize_text_field($data['prompt']);
    $save_request = new WP_REST_Request('POST', '/lucydream/v1/save-prompt');
    $save_request->set_param('prompt', $prompt);
    $save_result = save_spicy_prompt($save_request);
    if (is_wp_error($save_result)) {
    }
    // Commit credits and update usage
    if (isset($transaction_id)) {
        $commit_response = lucydream_commit_pending_credits($user_id, $price, $transaction_id, "Prompt generation ($prompt_type)");
        if (is_wp_error($commit_response) || !$commit_response['success']) {
            lucydream_revert_pending_credits($user_id, $price, $transaction_id, "Failed to commit $prompt_type prompt credits");
            return new WP_Error('credit_error', 'Failed to commit credits.', array('status' => 500));
        }
    }
    mm_update_used_prompts($user_id, $prompt_type);

    return new WP_REST_Response(array('prompt' => $prompt), 200);
}

/**
 * Register REST API routes for prompt generation.
 */
add_action('rest_api_init', function () {
    register_rest_route('lucydream/v1', '/save-prompt', array(
        'methods' => 'POST',
        'callback' => 'save_spicy_prompt',
        'permission_callback' => function($request) {
            return is_user_logged_in();
        },
        'args' => array(
            'prompt' => array('required' => true, 'type' => 'string')
        )
    ));

    register_rest_route('lucydream/v1', '/generate-prompt', array(
        'methods' => 'POST',
        'callback' => 'generate_spicy_prompt',
        'permission_callback' => function() { return is_user_logged_in(); },
    ));
});