console.log('lucydream-generate.js loaded successfully');

document.addEventListener('DOMContentLoaded', function () {
    console.log('DOMContentLoaded event fired successfully, now waiting for elements...');
    
    // Start periodic nonce refresh
    refreshNoncePeriodically();

    function waitForElements(maxAttempts = 50) {
        return new Promise((resolve, reject) => {
            let attempts = 0;
            const checkElements = setInterval(() => {
                const form = document.getElementById('custom-form');
                const message = document.getElementById('prompt-limit-message');
                const generateButton = document.getElementById('generate-button');
                const wrapper = document.getElementById('slider-wrapper');
                attempts++;
                console.log(`Attempt ${attempts}: Checking for elements - Form: ${!!form}, Message: ${!!message}, Button: ${!!generateButton}, Wrapper: ${!!wrapper}`);
                if (form && message && generateButton && wrapper) {
                    clearInterval(checkElements);
                    console.log('All required elements found successfully.');
                    resolve({ form, message, generateButton, wrapper });
                } else if (attempts >= maxAttempts) {
                    clearInterval(checkElements);
                    console.error('Elements not found after max attempts. Check if IDs match in HTML.');
                    reject(new Error('Required elements not found'));
                }
            }, 200);
        });
    }

    // Buy Credits form validation
    const buyCreditsForm = document.getElementById('buy-credits-form');
    if (buyCreditsForm) {
        buyCreditsForm.addEventListener('submit', function(e) {
            const creditsInput = document.getElementById('credits_amount');
            const message = document.getElementById('buy-credits-message');
            const credits = parseFloat(creditsInput.value);
            
            if (isNaN(credits) || credits < 1) {
                e.preventDefault();
                message.textContent = 'Please enter at least 1 credit.';
                return;
            }
            
            message.textContent = 'Processing your purchase...';
        });
    }

    // Periodic nonce refresh (every 6 hours)
    function refreshNoncePeriodically() {
        setInterval(async () => {
            try {
                const nonceResponse = await fetch('/wp-admin/admin-ajax.php?action=get_fresh_nonce', {
                    method: 'POST',
                    credentials: 'include',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                        'X-WP-Nonce': lucydream_data.nonce
                    },
                });
                const nonceData = await nonceResponse.json();
                if (nonceResponse.ok && nonceData.success && nonceData.data.nonce) {
                    lucydream_data.nonce = nonceData.data.nonce;
                    console.log('Nonce refreshed:', lucydream_data.nonce);
                } else {
                    console.warn('Periodic nonce refresh failed:', nonceData);
                }
            } catch (error) {
                console.error('Periodic nonce refresh error:', error);
            }
        }, 6 * 60 * 60 * 1000); // 6 hours
    }

    // Check user login status
    async function checkLoginStatus() {
        try {
            const response = await fetch('/wp-json/wp/v2/users/me', {
                headers: { 'X-WP-Nonce': lucydream_data.nonce },
                credentials: 'include',
            });
            if (!response.ok) {
                console.error('User not logged in or session expired:', response.status);
                message.textContent = 'Please log in again to generate images.';
                message.className = 'error';
                generateButton.disabled = true;
                return false;
            }
            return true;
        } catch (error) {
            console.error('Error checking login status:', error);
            message.textContent = 'Error verifying session. Please refresh the page.';
            message.className = 'error';
            generateButton.disabled = true;
            return false;
        }
    }

    waitForElements().then(({ form, message, generateButton, wrapper }) => {
        // Check if lucydream_data.nonce is defined
        if (!lucydream_data || !lucydream_data.nonce) {
            console.error('lucydream_data.nonce is not defined. Ensure wp_localize_script is configured correctly.');
            const loginUrl = window.location.origin + '/login/';
            message.innerHTML = `Error: Unable to initialize request. Please <a href="${loginUrl}">log in again</a>.`;
            message.className = 'error';
            generateButton.disabled = true;
            return;
        }

        console.log('Elements resolved successfully, setting up form handlers...');
        let slideIndex = 0;
        let lightboxIndex = 0;
        window.slides = window.slides || [];
        console.log('lucydream_data:', lucydream_data);
        let isInitialLoad = true;

        function cleanPrompt(fullPrompt, isSpicy = false) {
    let cleaned = fullPrompt;
    const triggerNames = lucydream_data.trigger_names || [];
    const classNames = lucydream_data.class_names || [];
    console.log('cleanPrompt called with prompt:', fullPrompt);
    console.log('Trigger names:', triggerNames);
    console.log('Class names:', classNames);
    
    const wordsToRemove = new Set([...triggerNames, ...classNames]);
    wordsToRemove.forEach(word => {
        const escaped = word.replace(/[-/\\^$*+?.()|[\]{}]/g, '\\$&');
        const regex = new RegExp(`\\b${escaped}\\b,?\\s*`, 'gi');
        const before = cleaned;
        cleaned = cleaned.replace(regex, '');
        if (before !== cleaned) {
            console.log(`Removed "${word}" from prompt: ${cleaned}`);
        } else {
            console.log(`No match for "${word}" in prompt: ${cleaned}`);
        }
    });

cleaned = cleaned.replace(/\bohwx\b,?\s*/gi, '').replace(/ohwx (woman|man),?\s*/gi, '');
console.log('After removing standalone ohwx and ohwx woman/man:', cleaned);

    if (isSpicy) {
        cleaned = cleaned.replace(/<lora:[^:]+:[0-1](\.\d)?>/gi, '');
        console.log('After removing LoRA tags (spicy mode):', cleaned);
    }

    cleaned = cleaned.trim().replace(/\s*,\s*/g, ', ').replace(/\s+/g, ' ');
    console.log('Final cleaned prompt:', cleaned);
    return cleaned;
}

        function generateDeviceId() {
            const deviceId = [
                navigator.userAgent.replace(/\s/g, ''),
                `${screen.width}x${screen.height}x${screen.colorDepth}`,
                navigator.platform || 'unknown',
                navigator.language || 'unknown',
                Intl.DateTimeFormat().resolvedOptions().timeZone || 'unknown',
                (function() {
                    const canvas = document.createElement('canvas');
                    const ctx = canvas.getContext('2d');
                    ctx.textBaseline = "top";
                    ctx.font = "14px 'Arial'";
                    ctx.fillStyle = "#f60";
                    ctx.fillRect(125, 1, 62, 20);
                    ctx.fillStyle = "#069";
                    ctx.fillText("Hello, World!", 2, 15);
                    ctx.fillStyle = "rgba(102, 204, 0, 0.7)";
                    ctx.fillText("Hello, World!", 4, 17);
                    return canvas.toDataURL();
                })(),
                (function() {
                    try {
                        const gl = document.createElement('canvas').getContext('webgl');
                        if (gl) {
                            const debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
                            return (debugInfo && (gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL) || gl.getParameter(debugInfo.UNMASKED_VENDOR_WEBGL))) || gl.getParameter(gl.RENDERER);
                        }
                        return 'no-webgl';
                    } catch (e) {
                        return 'webgl-error';
                    }
                })(),
                (function() {
                    try {
                        const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
                        const oscillator = audioCtx.createOscillator();
                        oscillator.type = 'sine';
                        oscillator.frequency.setValueAtTime(440, audioCtx.currentTime);
                        oscillator.start();
                        oscillator.stop();
                        return audioCtx.sampleRate + '|' + audioCtx.destination.numberOfChannels;
                    } catch (e) {
                        return 'audio-error';
                    }
                })(),
                navigator.hardwareConcurrency || 'unknown',
                Date.now().toString()
            ].join('|');
            return btoa(deviceId).substring(0, 64);
        }

        function setCookie(name, value, days) {
            const expires = new Date();
            expires.setTime(expires.getTime() + (days * 24 * 60 * 60 * 1000));
            document.cookie = `${name}=${value};expires=${expires.toUTCString()};path=/;SameSite=Strict`;
        }

        function getCookie(name) {
            const nameEQ = name + "=";
            const ca = document.cookie.split(';');
            for (let i = 0; i < ca.length; i++) {
                let c = ca[i];
                while (c.charAt(0) === ' ') c = c.substring(1, c.length);
                if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
            }
            return null;
        }

        function setLocalStorage(name, value) {
            try {
                localStorage.setItem(name, value);
            } catch (e) {
                console.warn('localStorage not available, falling back to cookies:', e);
                setCookie(name, value, 365);
            }
        }

        function getLocalStorage(name) {
            try {
                return localStorage.getItem(name);
            } catch (e) {
                console.warn('localStorage not available, falling back to cookies:', e);
                return getCookie(name);
            }
        }

        function setPersistentCount(deviceId, count) {
            setCookie(`custom_generations_${deviceId}`, count, 365);
            setLocalStorage(`custom_generations_${deviceId}`, count);
            try {
                sessionStorage.setItem(`custom_generations_${deviceId}`, count);
            } catch (e) {
                console.warn('sessionStorage not available, using cookies:', e);
            }
        }

        function getPersistentCount(deviceId) {
            const cookieCount = parseInt(getCookie(`custom_generations_${deviceId}`) || '0');
            const localCount = parseInt(getLocalStorage(`custom_generations_${deviceId}`) || '0');
            let sessionCount = 0;
            try {
                sessionCount = parseInt(sessionStorage.getItem(`custom_generations_${deviceId}`) || '0');
            } catch (e) {
                console.warn('sessionStorage not available, using cookies:', e);
            }
            return Math.max(cookieCount, localCount, sessionCount);
        }

        function setPersistentRequest(request) {
            const requestStr = JSON.stringify(request);
            setCookie('generation_request', requestStr, 1);
            try {
                localStorage.setItem('generation_request', requestStr);
            } catch (e) {
                console.warn('localStorage not available, using cookies:', e);
            }
        }

        function getPersistentRequest() {
            const cookieRequest = getCookie('generation_request');
            let localRequest = null;
            try {
                localRequest = localStorage.getItem('generation_request');
            } catch (e) {
                console.warn('localStorage not available, falling back to cookies:', e);
            }
            const requestStr = localRequest || cookieRequest;
            return requestStr ? JSON.parse(requestStr) : null;
        }

        function clearPersistentRequest() {
            document.cookie = 'generation_request=;expires=Thu, 01 Jan 1970 00:00:00 UTC;path=/;SameSite=Strict';
            try {
                localStorage.removeItem('generation_request');
            } catch (e) {
                console.warn('localStorage not available, cleared cookie:', e);
            }
        }

        const deviceIdInput = form.querySelector('#deviceId');
        if (deviceIdInput && !deviceIdInput.value) {
            deviceIdInput.value = generateDeviceId();
        }

        form.addEventListener('submit', async function (e) {
            console.log('Form submit event captured successfully! Preventing default refresh...');
            e.preventDefault();
            if (!await checkLoginStatus()) return;

            const promptInput = form.querySelector('#prompt-input');
            const modelInput = form.querySelector('#model');
            const userModelInput = form.querySelector('#user-model');
            const deviceId = deviceIdInput.value;
            let prompt = promptInput.value.trim();
            const selectedModel = modelInput ? modelInput.value : lucydream_data.default_tune_id;
            const model = selectedModel;
            const userModel = (selectedModel === 'my_models' && userModelInput) ? userModelInput.value : '';

            if (!prompt) {
                alert('Please enter a prompt');
                return;
            }
            
            generateButton.disabled = true;
            message.textContent = 'Generating...';
            message.className = 'generating';

            const generationRequest = { 
                prompt, 
                model,
                deviceId, 
                seed: -1,
                timestamp: Date.now()
            };
            if (userModel) generationRequest.user_model = userModel;
            setPersistentRequest(generationRequest);

            try {
                let freshNonce = lucydream_data.nonce;
                console.log('Attempting to fetch fresh nonce with initial nonce:', lucydream_data.nonce);
                try {
                    const nonceResponse = await fetch('/wp-admin/admin-ajax.php?action=get_fresh_nonce', {
                        method: 'POST',
                        credentials: 'include',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                            'X-WP-Nonce': lucydream_data.nonce
                        },
                    });
                    console.log('Nonce fetch response status:', nonceResponse.status);
                    const nonceText = await nonceResponse.text();
                    console.log('Nonce fetch response text:', nonceText);
                    let nonceData;
                    try {
                        nonceData = JSON.parse(nonceText);
                    } catch (jsonError) {
                        console.error('Nonce parse error:', jsonError);
                        throw new Error('Invalid nonce response format');
                    }
                    if (nonceResponse.ok && nonceData.success && nonceData.data.nonce) {
                        freshNonce = nonceData.data.nonce;
                        console.log('Fresh nonce obtained:', freshNonce);
                    } else {
                        console.warn('Nonce fetch failed:', nonceData);
                        throw new Error('Failed to fetch fresh nonce');
                    }
                } catch (nonceError) {
                    console.error('Nonce fetch error:', nonceError);
                    message.textContent = 'Error: Unable to verify request. Please refresh the page.';
                    message.className = 'error';
                    generateButton.disabled = false;
                    clearPersistentRequest();
                    return;
                }

                console.log('Sending request with headers:', {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': freshNonce,
                    'X-Request-Type': 'api'
                });
                console.log('Starting fetch request to API:', lucydream_data.api_url + 'generate');
                const response = await fetch(lucydream_data.api_url + 'generate', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': freshNonce,
                        'X-Request-Type': 'api'
                    },
                    credentials: 'include',
                    body: JSON.stringify(generationRequest),
                });
                console.log('Fetch response received:', response.status);

                const responseText = await response.text();
                console.log('API response text:', responseText);
                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (jsonError) {
                    console.error('JSON parse error:', jsonError);
                    if (response.status === 504) {
                        message.textContent = 'Server timed out. Image may still be generated—refresh to check.';
                        message.className = '';
                    } else {
                        message.textContent = 'Server error occurred. Please try again.';
                        message.className = '';
                    }
                    clearPersistentRequest();
                    return;
                }
                console.log('API response data:', data);

                if (response.status === 202) {
                    message.textContent = data.message || 'Image generation in progress. Please refresh in a few seconds.';
                    message.className = '';
                    const maxPollDuration = 5 * 60 * 1000; // 5 minutes
                    const pollStartTime = Date.now();
                    const pollInterval = setInterval(async () => {
                        if (Date.now() - pollStartTime > maxPollDuration) {
                            clearInterval(pollInterval);
                            message.textContent = 'Image generation timed out. Please try again.';
                            message.className = '';
                            clearPersistentRequest();
                            return;
                        }
                        try {
                            const pollResponse = await fetch(`/wp-json/lucydream/v1/fetch-task/${data.task_id || task_id}`, {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'X-WP-Nonce': freshNonce
                                },
                                credentials: 'include'
                            });
                            const pollData = await pollResponse.json();
                            if (pollResponse.ok && pollData.image_url) {
                                clearInterval(pollInterval);
                                const newImage = {
                                    image_url: pollData.image_url.startsWith('/wp-json') ? window.location.origin + pollData.image_url : pollData.image_url,
                                    thumbnail_url: pollData.thumbnail_url,
                                    prompt: pollData.prompt,
                                    user_prompt: pollData.user_prompt,
                                    timestamp: Date.now()
                                };
                                window.slides.unshift(newImage);
                                slideIndex = 0;
                                updateSlider();
                                message.textContent = '';
                                message.className = '';
                                promptInput.value = '';
                                const generationCount = getPersistentCount(deviceId);
                                setPersistentCount(deviceId, generationCount + 1);
                                clearPersistentRequest();
                                loadAllImages(true);
                            } else if (pollData.status === 'failed') {
                                clearInterval(pollInterval);
                                message.textContent = 'Image generation failed. Please try again.';
                                message.className = '';
                                clearPersistentRequest();
                            }
                        } catch (pollError) {
                            console.error('Polling error:', pollError);
                        }
                    }, 5000); // Poll every 5 seconds
                    return;
                }

                if (!response.ok) {
                    console.error(`Server returned ${response.status}:`, data);
                    if (response.status === 403 && data.error && data.error.includes('nonce')) {
                        message.textContent = 'Session expired. Please refresh the page and try again.';
                        message.className = 'error';
                    } else if (response.status === 403 && data.error && data.error_class === 'insufficient-credits') {
                        message.innerHTML = data.error;
                        message.className = data.error_class;
                    } else {
                        message.textContent = data.error || 'Something went wrong.';
                        message.className = data.error_class || '';
                    }
                    clearPersistentRequest();
                    return;
                }

                if (data.image_url) {
                    const newImage = {
                        image_url: data.image_url.startsWith('/wp-json') ? window.location.origin + data.image_url : data.image_url,
                        thumbnail_url: data.thumbnail_url,
                        prompt: data.prompt,
                        user_prompt: data.user_prompt,
                        timestamp: Date.now()
                    };
                    window.slides.unshift(newImage);
                    slideIndex = 0;
                    updateSlider();
                    message.textContent = '';
                    message.className = '';
                    promptInput.value = '';
                    const generationCount = getPersistentCount(deviceId);
                    setPersistentCount(deviceId, generationCount + 1);
                    clearPersistentRequest();
                    
                    setTimeout(() => {
                        loadAllImages(true);
                    }, 1000);
                } else {
                    message.textContent = 'Unexpected response from server.';
                    message.className = '';
                    clearPersistentRequest();
                }
            } catch (error) {
                console.error('Error during image generation - Full details:', error);
                message.textContent = error.message || 'Error connecting to the server.';
                message.className = '';
                clearPersistentRequest();
            } finally {
                generateButton.disabled = false;
            }
        });

        const modelElement = document.getElementById('model');
        if (modelElement) {
            modelElement.addEventListener('change', function () {
                console.log('Model changed, reloading images...');
                isInitialLoad = false;
                slideIndex = 0;
                loadAllImages();
            });
        }

        const userModelSelect = document.getElementById('user-model');
        if (userModelSelect) {
            userModelSelect.addEventListener('change', function () {
                console.log('User model changed, reloading images...');
                isInitialLoad = false;
                slideIndex = 0;
                loadAllImages();
            });
        }

        async function loadAllImages(forceRefresh = false) {
            try {
        const modelInput = document.getElementById('model');
        const userModelInput = document.getElementById('user-model');
        const selectedModel = modelInput ? modelInput.value : lucydream_data.default_tune_id;
        const modelParam = selectedModel;
        const userModelParam = (selectedModel === 'my_models' && userModelInput) ? userModelInput.value : '';
        const userId = lucydream_data.user_id;

        if (!modelParam) {
            console.log('No valid model selected, displaying placeholder message.');
            wrapper.innerHTML = '<p>Please select a model from the "Model:" dropdown</p>';
            window.slides = [];
            return;
        }

        if (modelParam === 'my_models' && !userModelParam) {
            console.log('My Models selected but no user model chosen yet.');
            wrapper.innerHTML = '<p>Please select a model from the "Name:" dropdown.</p>';
            return;
        }

        let url = `/wp-json/lucydream/v1/all-images?model=${modelParam}&user_id=${userId}`;
if (forceRefresh) {
    url += `&cache_bust=${Date.now()}`;
}
        if (userModelParam) url += `&model=${encodeURIComponent(userModelParam)}`;
if (forceRefresh) {
    url += `&cache_bust=${Date.now()}`;
}

        console.log('Fetching user images from URL:', url);
        const userResponse = await fetch(url, {
            cache: forceRefresh ? 'no-cache' : 'default',
            headers: { 'Cache-Control': forceRefresh ? 'no-cache' : 'public, max-age=300' }
        });
        const userData = await userResponse.json();
        console.log('User images response:', userData);

        let userImages = [];
        const persistentRequest = getPersistentRequest();
        if (userResponse.ok && Array.isArray(userData)) {
            userImages = userData.map(item => {
                if (item.user_prompt === 'Generated image' && persistentRequest && persistentRequest.deviceId === deviceIdInput.value) {
                    return {
                        image_url: item.image_url.startsWith('/wp-json') ? window.location.origin + item.image_url : item.image_url,
                        thumbnail_url: item.thumbnail_url,
                        prompt: item.prompt,
                        user_prompt: persistentRequest.prompt,
                        timestamp: Date.now()
                    };
                }
                return {
                    image_url: item.image_url.startsWith('/wp-json') ? window.location.origin + item.image_url : item.image_url,
                    thumbnail_url: item.thumbnail_url,
                    prompt: item.prompt,
                    user_prompt: item.user_prompt,
                    timestamp: item.timestamp || 0
                };
            });
            userImages.sort((a, b) => b.timestamp - a.timestamp);
        }

        window.slides = userImages.slice(0, 4);
        if (window.slides.length > 0) {
            updateSlider();
        } else {
            wrapper.innerHTML = '<p>No images generated yet.</p>';
        }
    } catch (error) {
        console.error('Error loading all images:', error);
        wrapper.innerHTML = '<p>Error loading images. Please refresh or try again later.</p>';
    }
}

        function updateSlider() {
            wrapper.innerHTML = '';
            console.log('Updating slider with slides:', window.slides);
            if (window.slides.length === 0) {
                wrapper.innerHTML = '<p>No images generated yet.</p>';
                return;
            }
            window.slides.forEach((slide, index) => {
                const item = document.createElement('div');
                item.className = 'slider-item';
                const img = document.createElement('img');
                img.src = slide.thumbnail_url;
                img.alt = slide.user_prompt;
                img.loading = 'lazy';
                img.onerror = function() { console.error('Thumbnail failed to load:', img.src); };
                item.appendChild(img);
                item.addEventListener('click', () => showLightbox(index));
                wrapper.appendChild(item);
            });
            wrapper.style.transform = `translateX(-${slideIndex * 220}px)`;
            document.getElementById('prev-slide').disabled = window.slides.length <= 1;
            document.getElementById('next-slide').disabled = window.slides.length <= 1;
        }

        function showLightbox(index) {
    lightboxIndex = index;
    const lightbox = document.getElementById('lightbox');
    const imageDiv = document.querySelector('.lightbox-image');
    const promptDiv = document.querySelector('.lightbox-prompt');
    const img = document.createElement('img');
    img.src = window.slides[index].image_url;
    img.alt = window.slides[index].user_prompt;
    img.loading = 'lazy';
    img.onerror = function() { console.error('Lightbox image failed to load:', img.src); };
    imageDiv.innerHTML = '';
    imageDiv.appendChild(img);

    const apiMode = document.getElementById('api_mode')?.value || 'spicy';
    let displayPrompt = cleanPrompt(window.slides[index].user_prompt, apiMode === 'spicy');
    
    // Ensure standalone 'ohwx' is explicitly removed for lightbox display
    displayPrompt = displayPrompt.replace(/\bohwx\b,?\s*/gi, '').trim();
    console.log('Final lightbox prompt after additional ohwx removal:', displayPrompt);

    promptDiv.innerHTML = '<strong>Prompt:</strong>';
    promptDiv.appendChild(document.createTextNode(` ${displayPrompt || 'Generated Image'}`));
    lightbox.style.display = 'flex';
    updateLightboxNav();
}

        function updateLightboxNav() {
            document.getElementById('lightbox-prev').disabled = window.slides.length <= 1;
            document.getElementById('lightbox-next').disabled = window.slides.length <= 1;
        }

        document.getElementById('prev-slide').addEventListener('click', () => {
            slideIndex = slideIndex > 0 ? slideIndex - 1 : window.slides.length - 1;
            updateSlider();
        });

        document.getElementById('next-slide').addEventListener('click', () => {
            slideIndex = slideIndex < window.slides.length - 1 ? slideIndex + 1 : 0;
            updateSlider();
        });

        document.getElementById('lightbox-prev').addEventListener('click', () => {
            lightboxIndex = lightboxIndex > 0 ? lightboxIndex - 1 : window.slides.length - 1;
            showLightbox(lightboxIndex);
        });

        document.getElementById('lightbox-next').addEventListener('click', () => {
            lightboxIndex = lightboxIndex < window.slides.length - 1 ? lightboxIndex + 1 : 0;
            showLightbox(lightboxIndex);
        });

        document.getElementById('lightbox').addEventListener('click', function(event) {
            if (!event.target.closest('.lightbox-content')) {
                document.getElementById('lightbox').style.display = 'none';
            }
        });

        console.log('Initial load completed, fetching latest images...');
        loadAllImages(true);
    }).catch(error => {
        console.error('waitForElements error:', error);
        message.textContent = 'Error loading form. Please refresh the page.';
        message.className = 'error';
    });
});