<?php
/*
Plugin Name: LucyDream Create Model
Description: Handles model creation and status updates as part of the main LucyDream plugin.
Version: 1.0
Author: LucyDream
*/


register_activation_hook(__FILE__, 'lucydream_create_model_activate');

function lucydream_create_model_activate() {
    if (null === get_page_by_path('create-model')) {
        $page_args = array(
            'post_title'   => 'Create Model',
            'post_name'    => 'create-model',
            'post_content' => '[user_info]',
            'post_content' => '[create_model_form]',
            'post_status'  => 'publish',
            'post_type'    => 'page',
        );
        wp_insert_post($page_args);
    }
}

// Ensure this file is only loaded within WordPress
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

require_once WP_CONTENT_DIR . '/plugins/aws-sdk/aws-autoloader.php';
use Aws\S3\S3Client;


/**
 * AJAX Handler for Create Model
 */
add_action('wp_ajax_create_model', 'handle_create_model');
function handle_create_model() {
    check_ajax_referer('wp_rest', '_wpnonce');
    if (!is_user_logged_in()) wp_send_json_error(['message' => 'You must be logged in'], 401);
    $user_id = get_current_user_id();
    $model_name = sanitize_text_field($_POST['model_name']);
    $random_digits = sprintf("%03d", rand(0, 999));
    $model_name_with_digits = "{$model_name}-{$random_digits}";
    $safe_model_name = str_replace(' ', '-', $model_name_with_digits);
    $full_model_name = "{$model_name_with_digits}-{$user_id}";
    $gender = sanitize_text_field($_POST['gender']);
    $device_id = sanitize_text_field($_POST['deviceId']);
    if (empty($model_name)) wp_send_json_error(['message' => 'Model name is required'], 400);
    if (strlen($model_name) > 15) wp_send_json_error(['message' => 'Model name must be 15 characters or less'], 400);
    $image_count = 0;
    $allowed_formats = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp'];
    for ($i = 1; $i <= 5; $i++) {
        if (!empty($_FILES["image_$i"]['name'])) {
            $file_type = $_FILES["image_$i"]['type'];
            if (in_array($file_type, $allowed_formats)) $image_count++;
            else wp_send_json_error(['message' => "Image $i must be in JPEG, PNG, or WEBP format"], 400);
        }
    }
    if ($image_count !== 5) wp_send_json_error(['message' => 'Exactly 5 images (JPEG, PNG, or WEBP) are required'], 400);
    $credit_cost = get_option('mm_model_creation_cost', 6.0);
    global $wpdb;
    $wpdb->query('START TRANSACTION');
    $balance = lucydream_calculate_balance($user_id);
    if ($balance < $credit_cost) {
    $wpdb->query('ROLLBACK');
    wp_send_json_error([
        'message' => 'Model training requires ' . number_format($credit_cost, 2) . ' credits. <a href="/payments/" class="buy-credits-link" target="_blank">Buy more credits</a>',
        'message_class' => 'insufficient-credits'
    ], 403);
}
    $hold_response = lucydream_hold_credits($user_id, $credit_cost, 'Model training');
if (is_wp_error($hold_response) || !isset($hold_response['success']) || !$hold_response['success']) {
    $wpdb->query('ROLLBACK');
    $error_msg = is_wp_error($hold_response) ? $hold_response->get_error_message() : ($hold_response['message'] ?? 'Unknown error');
    wp_send_json_error(['message' => "Credit hold failed: $error_msg"], 500);
}
$transaction_id = $hold_response['transaction_id'];

    $lock_key = "lucydream_model_lock_$user_id";
    $lock_timeout = 300;
    if (get_transient($lock_key)) {
        $wpdb->query('ROLLBACK');
        wp_send_json_error(['message' => 'Another model creation is in progress. Please try again later.'], 429);
    }
    set_transient($lock_key, true, $lock_timeout);

    $upload_dir = wp_upload_dir();
    $temp_dir = $upload_dir['basedir'] . "/custom-images/temp/$user_id/$safe_model_name/";
    if (!file_exists($temp_dir)) {
        if (!mkdir($temp_dir, 0755, true)) {
            delete_transient($lock_key);
            $wpdb->query('ROLLBACK');
            wp_send_json_error(['message' => 'Failed to create temporary directory'], 500);
        }
    }

    $image_urls = [];
    $unique_suffix = time() . '-' . $safe_model_name;
    for ($i = 1; $i <= 5; $i++) {
        $file = $_FILES["image_$i"];
        $temp_path = $temp_dir . "image_$i-$unique_suffix.jpg";
        if (move_uploaded_file($file['tmp_name'], $temp_path)) {
            $image_urls[] = $upload_dir['baseurl'] . "/custom-images/temp/$user_id/$safe_model_name/image_$i-$unique_suffix.jpg";
        } else {
            for ($j = 1; $j < $i; $j++) {
                $cleanup_path = $temp_dir . "image_$j-$unique_suffix.jpg";
                if (file_exists($cleanup_path)) unlink($cleanup_path);
            }
            delete_transient($lock_key);
            $wpdb->query('ROLLBACK');
            wp_send_json_error(['message' => "Failed to upload image $i"], 500);
        }
    }

$proxy_api_key = get_option('lucydream_proxy_secret_key', '');


    $url = 'https://my.lucydream.ai/wp-json/whitelabel/v1/generate';
    $data = [
        'tune' => [
            'title' => $full_model_name,
            'base_tune_id' => 1504944,
            'preset' => 'flux-lora-portrait',
            'model_type' => 'lora',
            'face_detection' => true,
            'name' => $gender,
            'token' => 'ohwx',
            'callback' => home_url('/wp-json/custom/v1/model-created-callback')
        ]
    ];
    $data['tune']['image_urls'] = $image_urls;
    set_time_limit(120);
    $response = wp_remote_post($url, [
        'headers' => ['X-Proxy-Key' => PROXY_SECRET_KEY, 'X-Request-Type'  => 'api', 'Content-Type' => 'application/json'],
        'body' => json_encode($data),
        'timeout' => 120
    ]);

    for ($i = 1; $i <= 5; $i++) {
        $temp_path = $temp_dir . "image_$i-$unique_suffix.jpg";
        if (file_exists($temp_path)) {
            unlink($temp_path);
        }
    }

    if (is_wp_error($response)) {
        delete_transient($lock_key);
        $wpdb->query('ROLLBACK');
        wp_send_json_error(['message' => 'Model creation service is temporarily unavailable. Please try again later.'], 503);
    }
    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    if ($status_code !== 201 && $status_code !== 200) {
        delete_transient($lock_key);
        $wpdb->query('ROLLBACK');
        $error_msg = 'Model creation service is temporarily unavailable. Please try again later.';
        if ($status_code === 503) {
        } else {
            $decoded_body = json_decode($body, true);
            $error_msg = isset($decoded_body['error']) ? $decoded_body['error'] : $error_msg;
        }
        wp_send_json_error(['message' => $error_msg], $status_code ?: 503);
    }
    $decoded_body = json_decode($body, true);
if (isset($decoded_body['data'])) $decoded_body = $decoded_body['data']; 
if (!isset($decoded_body['id']) && !isset($decoded_body['tune_id'])) {
    delete_transient($lock_key);
    $wpdb->query('ROLLBACK');
    wp_send_json_error(['message' => 'Failed to initiate model creation: Invalid response from service'], 500);
}
$tune_id = $decoded_body['id'] ?? $decoded_body['tune_id']; 

    if (function_exists('mm_add_user_model')) {
        mm_add_user_model([
            'name' => $model_name_with_digits,
            'tune_id' => $tune_id,
            'user_id' => $user_id,
            'type' => 'user_flux',
            'credit_cost' => 0.00,
            'class_name' => $gender,
            'training_status' => 'pending',
            'full_name' => $full_model_name
        ]);
    }
    $wpdb->query('COMMIT');
    delete_transient($lock_key);
    wp_send_json_success(['message' => 'Model creation started', 'tune_id' => $tune_id]);
}

/**
 * AJAX Handler for Update Model Status
 */
add_action('wp_ajax_update_model_status', 'update_model_status_handler');
function update_model_status_handler() {
    if (!is_user_logged_in()) wp_send_json_error(['message' => 'Not logged in'], 401);
    check_ajax_referer('wp_rest', '_wpnonce');
    $tune_id = sanitize_text_field($_POST['tune_id']);
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    $wpdb->update($table_name, ['training_status' => 'ready'], ['tune_id' => $tune_id, 'user_id' => get_current_user_id()]);
    wp_send_json_success(['message' => 'Model status updated']);
}

/**
 * AJAX Handler for Model Images
 */
add_action('wp_ajax_get_model_images', 'get_model_images_handler');
add_action('wp_ajax_nopriv_get_model_images', 'get_model_images_handler');
function get_model_images_handler() {
    $tune_id = sanitize_text_field($_POST['tune_id'] ?? '');
    $celeb = sanitize_text_field($_POST['celeb'] ?? '');
    $upload_dir = wp_upload_dir();
    $model_dir = $upload_dir['basedir'] . '/custom-images/models-page/';
    $model_url = $upload_dir['baseurl'] . '/custom-images/models-page/';
    if (!is_dir($model_dir)) { wp_send_json_success(['images' => []]); exit; }
    $files = glob($model_dir . '*.jpg');
    $images = [];
    foreach ($files as $file) {
        $basename = basename($file);
        $parts = explode('_', str_replace('.jpg', '', $basename));
        $file_tune_id = array_pop($parts);
        $is_thumbnail = $file_tune_id === 'thumb';
        if ($is_thumbnail) $file_tune_id = array_pop($parts);
        if ($file_tune_id === $tune_id && !$is_thumbnail) {
            $thumb_file = $model_dir . implode('_', $parts) . '_' . $file_tune_id . '_thumb.jpg';
            $images[] = [
                'full_url' => $model_url . $basename,
                'thumb_url' => file_exists($thumb_file) ? $model_url . basename($thumb_file) : $model_url . $basename,
                'prompt' => 'Model Image'
            ];
        }
    }
    wp_send_json_success(['images' => $images]); wp_die();
}

/**
 * AJAX Endpoint to Fetch a Fresh Nonce After Login
 */
add_action('wp_ajax_get_fresh_nonce', 'get_fresh_nonce');
add_action('wp_ajax_nopriv_get_fresh_nonce', 'get_fresh_nonce');
function get_fresh_nonce() {
    if (!is_user_logged_in()) {
        wp_send_json_error(['message' => 'Not logged in'], 401);
    }
    wp_send_json_success(['nonce' => wp_create_nonce('wp_rest')]);
}


/**
 * Check pending models and update status via cron
 */
function check_pending_models() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    
    $cutoff_time = date('Y-m-d H:i:s', strtotime('-10 minutes'));
    $pending_models = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT tune_id, user_id, created_at, training_status 
             FROM $table_name 
             WHERE training_status = 'pending' 
             AND created_at < %s",
            $cutoff_time
        )
    );

    if (empty($pending_models)) {
        return;
    }
    
    // Initialize R2/S3 client (credentials from admin settings)
$account_id = get_option('lucydream_r2_account_id', '');
$endpoint = "https://{$account_id}.r2.cloudflarestorage.com";
$public_endpoint = get_option('lucydream_bucket_url', "https://{$account_id}.r2.cloudflarestorage.com");
$s3 = new S3Client([
    'region' => 'auto',
    'endpoint' => $endpoint,
    'credentials' => [
        'key' => get_option('lucydream_r2_access_key_id', ''),
        'secret' => get_option('lucydream_r2_secret_key', '')
    ],
    'version' => 'latest'
]);
$bucket = get_option('lucydream_bucket_name', '');

    foreach ($pending_models as $model) {
        $tune_id = $model->tune_id;
        $user_id = $model->user_id;
        
        // Direct R2 check
try {
    $prefix = ''; 
    $result = $s3->listObjectsV2([
        'Bucket' => $bucket,
        'Prefix' => $prefix,
    ]);
    
    $ckpt_urls = [];
    if (isset($result['Contents'])) {
        foreach ($result['Contents'] as $object) {
            $key = $object['Key'];
            if (strpos($key, $tune_id) !== false && str_ends_with(strtolower($key), '.safetensors')) {
                // Include /{$bucket}/ in the URL to match backend format
                $ckpt_urls[] = $public_endpoint . '/' . $bucket . '/' . $key;
            }
        }
    }
            
            if (!empty($ckpt_urls)) {
                $payload = ['id' => $tune_id, 'ckpt_urls' => $ckpt_urls];
                $callback_request = new WP_REST_Request('POST', '/custom/v1/model-created-callback');
                $callback_request->set_body(json_encode($payload));
                $callback_response = rest_do_request($callback_request);

                if ($callback_response->is_error()) {
                } else {
                    $new_status = $wpdb->get_var($wpdb->prepare(
                        "SELECT training_status FROM $table_name WHERE tune_id = %s",
                        $tune_id
                    ));
                }
            } else {
            }
        } catch (Exception $e) {
}
    }
}


add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/model-created-callback', array(
        'methods' => 'POST',
        'callback' => 'handle_model_created_callback',
        'permission_callback' => '__return_true'
    ));
    register_rest_route('custom/v1', '/trigger-pending-models', array(
        'methods' => 'POST',
        'callback' => 'trigger_pending_models_check',
        'permission_callback' => function($request) {
            $secret = get_option('lucydream_cron_secret_key');
            $provided_secret = $request->get_param('secret');
            if ($provided_secret !== $secret) {
                return new WP_Error('invalid_secret', 'Invalid secret', array('status' => 403));
            }
            return true;
        },
    ));
    register_rest_route('custom/v1', '/trigger-model-deletion', array(
        'methods' => 'POST',
        'callback' => 'trigger_model_deletion',
        'permission_callback' => function($request) {
            $secret = get_option('lucydream_cron_secret_key');
            $provided_secret = $request->get_param('secret');
            if ($provided_secret !== $secret) {
                return new WP_Error('invalid_secret', 'Invalid secret', array('status' => 403));
            }
            return true;
        },
    ));
});

function handle_model_created_callback($request) {
    $body = json_decode($request->get_body(), true);
    if (!isset($body['id']) || !isset($body['ckpt_urls'])) {
        return new WP_REST_Response(['error' => 'Invalid callback data'], 400);
    }
    $tune_id = $body['id'];
    $ckpt_urls = $body['ckpt_urls'];
    
    $cleaned_urls = array_map(function($url) {
        return str_replace('\/', '/', $url); 
    }, (array)$ckpt_urls);
    
    $safetensors_path = !empty($cleaned_urls) ? $cleaned_urls[0] : '';
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    $created_at = $wpdb->get_var($wpdb->prepare(
        "SELECT created_at FROM $table_name WHERE tune_id = %s",
        $tune_id
    ));
    
    if (!$created_at) {
        return new WP_REST_Response(['error' => 'Failed to fetch model creation date'], 500);
    }
    
    // Calculate safetensors_expires_at as created_at + 30 days
    $expires_at = date('Y-m-d H:i:s', strtotime($created_at . ' +30 days'));
    
    $update_result = $wpdb->update(
        $table_name,
        [
            'training_status' => 'ready',
            'safetensors_path' => $safetensors_path,
            'safetensors_expires_at' => $expires_at
        ],
        ['tune_id' => $tune_id],
        ['%s', '%s', '%s'],
        ['%s']
    );
    
    if ($update_result === false) {
        return new WP_REST_Response(['error' => 'Failed to update model status'], 500);
    }
    
    return new WP_REST_Response(['success' => true], 200);
}

function trigger_pending_models_check($request) {
    check_pending_models();
    return new WP_REST_Response(['success' => true, 'message' => 'Pending models check triggered'], 200);
}

function trigger_model_deletion($request) {
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'model_manager_user_models';
    $current_time = current_time('mysql');
    
    // Initialize R2/S3 client
    $account_id = get_option('lucydream_r2_account_id', '');
    $endpoint = "https://{$account_id}.r2.cloudflarestorage.com";
    $s3 = new S3Client([
        'region' => 'auto',
        'endpoint' => $endpoint,
        'credentials' => [
            'key' => get_option('lucydream_r2_access_key_id', ''),
            'secret' => get_option('lucydream_r2_secret_key', '')
        ],
        'version' => 'latest'
    ]);
    $bucket = get_option('lucydream_bucket_name', '');
    
    // Query for expired models
    $expired_models = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT tune_id, safetensors_path 
             FROM $table_name 
             WHERE safetensors_expires_at < %s",
            $current_time
        )
    );
    
    if (empty($expired_models)) {
        return new WP_REST_Response(['success' => true, 'message' => 'No expired models found'], 200);
    }
    
    foreach ($expired_models as $model) {
        $tune_id = $model->tune_id;
        $safetensors_path = $model->safetensors_path;
    
        $key = parse_url($safetensors_path, PHP_URL_PATH);
        $key = ltrim($key, '/'); 
        $key = str_replace($bucket . '/', '', $key); 
    
        if (!empty($key)) {
            try {
                $s3->deleteObject([
                    'Bucket' => $bucket,
                    'Key' => $key
                ]);
            } catch (Exception $e) {
                continue; 
            }
        } else {
        }
    
        // Delete from database
        $deleted = $wpdb->delete($table_name, ['tune_id' => $tune_id]);
        if ($deleted) {
        } else {
        }
    }
    
    return new WP_REST_Response(['success' => true, 'message' => 'Model deletion triggered'], 200);
}


/**
 * Shortcode for Create Model Form
 */
add_shortcode('create_model_form', 'lucydream_create_model_form');
function lucydream_create_model_form() {
    wp_enqueue_script('lucydream-create-model', plugin_dir_url(__FILE__) . 'lucydream-create-model.js', array(), '1.0', true);
    wp_localize_script('lucydream-create-model', 'lucydream_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('wp_rest')
    ));
    wp_enqueue_style('lucydream-create-model', plugin_dir_url(__FILE__) . 'lucydream-create-model.css', array(), '1.0');

    ob_start();
    ?>
    <div id="custom-container">
        <h2 style="margin-top: 50px;">Create Your Model</h2>
        <form class="custom-form" id="create-model-form" method="post" enctype="multipart/form-data" style="position: relative;">
            <input type="hidden" name="action" value="create_model">
            <div class="dropdown-container" style="display: flex; gap: 15px; align-items: center;">
                <span id="name-info">Name: <input type="text" id="model-name" name="model_name" required maxlength="15" placeholder="Enter model name"></span>
            </div>
            <div class="dropdown-container gender-container" style="display: flex; gap: 15px; align-items: center; margin-top: 20px; margin-bottom: 40px;">
                <span id="gender-info">Gender: 
                    <select id="gender" name="gender" required>
                        <option value="" selected disabled>---</option>
                        <option value="man">Man</option>
                        <option value="woman">Woman</option>
                    </select>
                </span>
            </div>
            <div id="image-upload-container" style="margin-top: 10px;">
                <h4>Upload 5 different images of the same person:</h4>
                <?php for ($i = 1; $i <= 5; $i++): ?>
                    <p class="field-block">
                        <label for="image_<?php echo $i; ?>">Image <?php echo $i; ?></label>
                        <input type="file" name="image_<?php echo $i; ?>" id="image_<?php echo $i; ?>" accept="image/jpeg,image/png,image/webp" required>
                    </p>
                <?php endfor; ?>
            </div>
            <input type="hidden" id="deviceId" name="deviceId">
            <p class="warning-text" style="margin-top: 40px;">⚠️ By creating a model, you confirm that you are either using images of yourself or have permission to use the images for this purpose. You agree to take responsibility for ensuring the images are used appropriately and in compliance with applicable laws.</p>
            <div style="display: flex; justify-content: space-between; align-items: center;">
              <button type="submit" id="create-model-button">Create A Model</button>
            </div>
            <p id="model-status-message"></p>
        </form>
    </div>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        function generateDeviceId() {
            return btoa([navigator.userAgent.replace(/\s/g, ''), `${screen.width}x${screen.height}x${screen.colorDepth}`, navigator.platform || 'unknown', Date.now()].join('|')).substring(0, 64);
        }
        document.getElementById('deviceId').value = generateDeviceId();
    });
    </script>
    <?php
    return ob_get_clean();
}
?>